﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.IO;
using Dolinay; // Imported class for drive insertion event handling

namespace DriveLogger
{
    public struct DriveEntry
    {
        public DateTime time;
        public string drive;
        public string label;
        public string size;
        public string owner;
    }
    public partial class MainForm : Form
    {
        private static List<DriveEntry> driveList = new List<DriveEntry>();
        private static string logLocation = "C:\\DriveLog.txt";
        public MainForm()
        {
            // Clears the list of DriveEntry structs
            driveList.Clear();
            InitializeComponent();
            DriveDetector driveDetector = new DriveDetector();
            driveDetector.DeviceArrived += new DriveDetectorEventHandler(driveDetector_DeviceArrived);
            driveDetector.DeviceRemoved += new DriveDetectorEventHandler(driveDetector_DeviceRemoved);
            this.listView_Drives.AfterLabelEdit += new LabelEditEventHandler(listView_Drives_AfterLabelEdit);
            this.listView_Drives.Validating += new CancelEventHandler(listView_Drives_Validating);

            this.KeyPreview = true;
            this.KeyPress += new KeyPressEventHandler(MainForm_KeyPress);
            
            using (StreamWriter sw = File.AppendText(logLocation))
            {
                sw.WriteLine("-- New Session Started --");
                sw.WriteLine("Initializing form details");
            }
            this.listView_Drives.Columns.Add("Owner", 145, HorizontalAlignment.Left);
            this.listView_Drives.Columns.Add("Time", 130, HorizontalAlignment.Center);
            this.listView_Drives.Columns.Add("Drive", 40, HorizontalAlignment.Center);
            this.listView_Drives.Columns.Add("Label", 109, HorizontalAlignment.Center);
            this.listView_Drives.Columns.Add("Size", 60, HorizontalAlignment.Center);

            paintDriveListbox();
        }

        void listView_Drives_Validating(object sender, CancelEventArgs e)
        {
            paintDriveListbox();
        }

        void listView_Drives_AfterLabelEdit(object sender, LabelEditEventArgs e)
        {
            if (e.Label != null)
            {
                using (StreamWriter sw = File.AppendText(logLocation))
                {
                    ListViewItem entry = listView_Drives.Items[e.Item];

                    sw.WriteLine("Label \"" + e.Label + "\" added to drive " + entry.SubItems[2].Text);
                }
            }
        }

        void MainForm_KeyPress(object sender, KeyPressEventArgs e)
        {
            switch (e.KeyChar)
            {
                case '?':
                    AboutBox window = new AboutBox();
                    window.ShowDialog();
                    break;
            }
        }
        private void paintDriveListbox()
        {
            this.listView_Drives.BeginUpdate();

            this.listView_Drives.Items.Clear();

            // Adds each entry from the driveList into the listView
            foreach (DriveEntry entry in driveList)
            {
                ListViewItem item = new ListViewItem();
                item.Text = entry.owner;
                ListViewItem.ListViewSubItem subTime = new ListViewItem.ListViewSubItem();
                subTime.Text = entry.time.ToString();
                item.SubItems.Add(subTime);
                ListViewItem.ListViewSubItem subDrive = new ListViewItem.ListViewSubItem();
                subDrive.Text = entry.drive;
                item.SubItems.Add(subDrive);
                ListViewItem.ListViewSubItem subLabel = new ListViewItem.ListViewSubItem();
                subLabel.Text = entry.label;
                item.SubItems.Add(subLabel);
                ListViewItem.ListViewSubItem subSize = new ListViewItem.ListViewSubItem();
                subSize.Text = entry.size;
                item.SubItems.Add(subSize);

                this.listView_Drives.Items.Add(item);
            }

            this.listView_Drives.EndUpdate();
        }
        void driveDetector_DeviceArrived(object sender, DriveDetectorEventArgs e)
        {
            e.HookQueryRemove = true;

            DriveEntry newEntry = new DriveEntry();
            newEntry.time = DateTime.Now;
            newEntry.drive = e.Drive;

            DriveInfo tempDrive = null;
            DriveInfo[] allDrives = DriveInfo.GetDrives();
            foreach (DriveInfo drive in allDrives)
            {
                if (drive.IsReady)
                {
                    if (drive.Name == newEntry.drive)
                    {
                        tempDrive = drive;
                        break;
                    }
                }
            }
            newEntry.label = tempDrive.VolumeLabel;
            newEntry.size = (tempDrive.TotalSize / 1073741824).ToString() + " GB";

            if (newEntry.label != "System Reserved")
            {
                LabelPrompt label = new LabelPrompt();
                label.ShowDialog();
                newEntry.owner = label.driveLabel;
                driveList.Add(newEntry);
            }

            using (StreamWriter sw = File.AppendText(logLocation))
            {
                sw.WriteLine("Drive Attached -- [" + newEntry.time.ToString() + "]\t\"" + newEntry.owner + "\"\t" + newEntry.drive + "\t\"" + newEntry.label + "\"\t" + newEntry.size);
            }

            paintDriveListbox();
        }
        void driveDetector_DeviceRemoved(object sender, DriveDetectorEventArgs e)
        {
            //DriveEntry entryToRemove = new DriveEntry();
            foreach (DriveEntry entry in driveList)
            {
                if (e.Drive == entry.drive)
                {
                    //entryToRemove = entry;
                    driveList.Remove(entry);
                    break;
                }
            }
            //driveList.Remove(entryToRemove);

            using (StreamWriter sw = File.AppendText(logLocation))
            {
                //sw.WriteLine("Drive Removed --  [" + entryToRemove.time.ToString() + "]\t" + entryToRemove.drive + "\t\"" + entryToRemove.label + "\"\t" + entryToRemove.size);
            }

            paintDriveListbox();
        }
    }
}
